/*
 * Decompiled with CFR 0.152.
 */
package org.apache.activemq.artemis.tests.integration.stomp;

import jakarta.jms.BytesMessage;
import jakarta.jms.Destination;
import jakarta.jms.Message;
import jakarta.jms.MessageConsumer;
import jakarta.jms.MessageProducer;
import jakarta.jms.TextMessage;
import java.io.ByteArrayOutputStream;
import java.lang.invoke.MethodHandles;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.HashSet;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.activemq.artemis.api.config.ActiveMQDefaultConfiguration;
import org.apache.activemq.artemis.api.core.JsonUtil;
import org.apache.activemq.artemis.api.core.QueueConfiguration;
import org.apache.activemq.artemis.api.core.RoutingType;
import org.apache.activemq.artemis.api.core.SimpleString;
import org.apache.activemq.artemis.api.core.client.ActiveMQClient;
import org.apache.activemq.artemis.api.core.client.ClientMessage;
import org.apache.activemq.artemis.api.core.client.ClientProducer;
import org.apache.activemq.artemis.api.core.client.ClientSession;
import org.apache.activemq.artemis.api.core.client.ClientSessionFactory;
import org.apache.activemq.artemis.api.core.client.ServerLocator;
import org.apache.activemq.artemis.api.core.management.ManagementHelper;
import org.apache.activemq.artemis.api.jms.ActiveMQJMSClient;
import org.apache.activemq.artemis.core.management.impl.ActiveMQServerControlImpl;
import org.apache.activemq.artemis.core.management.impl.view.ProducerField;
import org.apache.activemq.artemis.core.postoffice.Binding;
import org.apache.activemq.artemis.core.postoffice.QueueBinding;
import org.apache.activemq.artemis.core.postoffice.impl.LocalQueueBinding;
import org.apache.activemq.artemis.core.protocol.stomp.StompProtocolManager;
import org.apache.activemq.artemis.core.server.ActiveMQServer;
import org.apache.activemq.artemis.core.server.Queue;
import org.apache.activemq.artemis.core.server.impl.ActiveMQServerImpl;
import org.apache.activemq.artemis.core.server.impl.AddressInfo;
import org.apache.activemq.artemis.core.settings.impl.AddressSettings;
import org.apache.activemq.artemis.jms.client.ActiveMQMessage;
import org.apache.activemq.artemis.json.JsonArray;
import org.apache.activemq.artemis.json.JsonObject;
import org.apache.activemq.artemis.logs.AssertionLoggerHandler;
import org.apache.activemq.artemis.reader.MessageUtil;
import org.apache.activemq.artemis.spi.core.remoting.Acceptor;
import org.apache.activemq.artemis.tests.extensions.parameterized.ParameterizedTestExtension;
import org.apache.activemq.artemis.tests.integration.mqtt.FuseMQTTClientProvider;
import org.apache.activemq.artemis.tests.integration.stomp.StompTestBase;
import org.apache.activemq.artemis.tests.integration.stomp.util.ClientStompFrame;
import org.apache.activemq.artemis.tests.integration.stomp.util.StompClientConnection;
import org.apache.activemq.artemis.tests.integration.stomp.util.StompClientConnectionFactory;
import org.apache.activemq.artemis.tests.util.Wait;
import org.apache.activemq.artemis.utils.RandomUtil;
import org.apache.activemq.artemis.utils.collections.IterableStream;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.TestTemplate;
import org.junit.jupiter.api.extension.ExtendWith;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ExtendWith(value={ParameterizedTestExtension.class})
public class StompTest
extends StompTestBase {
    private static final Logger logger = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    protected StompClientConnection conn;

    @Override
    protected ActiveMQServer createServer() throws Exception {
        ActiveMQServer server = super.createServer();
        server.getConfiguration().setAddressQueueScanPeriod(100L);
        return server;
    }

    @Override
    @BeforeEach
    public void setUp() throws Exception {
        super.setUp();
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
    }

    @Override
    @AfterEach
    public void tearDown() throws Exception {
        try {
            boolean connected = this.conn != null && this.conn.isConnected();
            logger.debug("Connection 1.0 connected: {}", (Object)connected);
            if (connected) {
                try {
                    this.conn.disconnect();
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
        }
        finally {
            super.tearDown();
            this.conn.closeTransport();
        }
    }

    @TestTemplate
    public void testConnectionTTL() throws Exception {
        int port = 61614;
        URI uri = StompTest.createStompClientUri(this.scheme, this.hostname, port);
        this.server.getRemotingService().createAcceptor("test", "tcp://127.0.0.1:" + port + "?connectionTtl=1000").start();
        StompClientConnection conn = StompClientConnectionFactory.createClientConnection(uri);
        conn.connect("brianm", "wombats");
        Thread.sleep(5000L);
        ClientStompFrame frame = conn.receiveFrame();
        Assertions.assertEquals((Object)"ERROR", (Object)frame.getCommand());
        Assertions.assertFalse((boolean)conn.isConnected());
    }

    @TestTemplate
    public void testSendManyMessages() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        int count = 1000;
        CountDownLatch latch = new CountDownLatch(count);
        consumer.setMessageListener(arg0 -> latch.countDown());
        for (int i = 1; i <= count; ++i) {
            StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "Hello World!");
        }
        Assertions.assertTrue((boolean)latch.await(60L, TimeUnit.SECONDS));
    }

    @TestTemplate
    public void testProducerMetrics() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "Hello World!", true);
        String filterString = this.createJsonFilter("", "", "");
        String producersAsJsonString = this.server.getActiveMQServerControl().listProducers(filterString, 1, 50);
        JsonObject producersAsJsonObject = JsonUtil.readJsonObject((String)producersAsJsonString);
        JsonArray array = (JsonArray)producersAsJsonObject.get((Object)"data");
        Assertions.assertEquals((int)1, (int)array.size(), (String)"number of producers returned from query");
        JsonObject producer = array.getJsonObject(0);
        Assertions.assertEquals((int)1, (int)producer.getInt(ProducerField.MESSAGE_SENT.getName()), (String)ProducerField.MESSAGE_SENT.getName());
        StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "Hello World!", true);
        producersAsJsonString = this.server.getActiveMQServerControl().listProducers(filterString, 1, 50);
        producersAsJsonObject = JsonUtil.readJsonObject((String)producersAsJsonString);
        array = (JsonArray)producersAsJsonObject.get((Object)"data");
        Assertions.assertEquals((int)1, (int)array.size(), (String)"number of producers returned from query");
        producer = array.getJsonObject(0);
        Assertions.assertEquals((int)2, (int)producer.getInt(ProducerField.MESSAGE_SENT.getName()), (String)ProducerField.MESSAGE_SENT.getName());
        this.conn.closeTransport();
        Wait.assertEquals((int)0, () -> ((JsonArray)JsonUtil.readJsonObject((String)this.server.getActiveMQServerControl().listProducers(filterString, 1, 50)).get((Object)"data")).size());
        producersAsJsonString = this.server.getActiveMQServerControl().listProducers(filterString, 1, 50);
        producersAsJsonObject = JsonUtil.readJsonObject((String)producersAsJsonString);
        array = (JsonArray)producersAsJsonObject.get((Object)"data");
        Assertions.assertEquals((int)0, (int)array.size(), (String)"number of producers returned from query");
    }

    @TestTemplate
    public void testSendOverDiskFull() throws Exception {
        try (AssertionLoggerHandler loggerHandler = new AssertionLoggerHandler();){
            MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
            this.conn.connect(this.defUser, this.defPass);
            int count = 1000;
            CountDownLatch latch = new CountDownLatch(count);
            consumer.setMessageListener(arg0 -> latch.countDown());
            ((ActiveMQServerImpl)this.server).getMonitor().setMaxUsage(0.0).tick();
            Exception e = null;
            try {
                for (int i = 1; i <= count; ++i) {
                    StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "Hello World!");
                }
            }
            catch (Exception se) {
                e = se;
            }
            Assertions.assertNotNull((Object)e);
            loggerHandler.findText(new String[]{"AMQ119119"});
        }
    }

    @TestTemplate
    public void testConnect() throws Exception {
        ClientStompFrame frame = this.conn.createFrame("CONNECT").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("request-id", "1");
        ClientStompFrame response = this.conn.sendFrame(frame);
        Assertions.assertTrue((boolean)response.getCommand().equals("CONNECTED"));
        Assertions.assertTrue((boolean)response.getHeader("response-id").equals("1"));
    }

    @TestTemplate
    public void testDisconnectAndError() throws Exception {
        ClientStompFrame frame = this.conn.createFrame("CONNECT").addHeader("login", this.defUser).addHeader("passcode", this.defPass).addHeader("request-id", "1");
        ClientStompFrame response = this.conn.sendFrame(frame);
        Assertions.assertTrue((boolean)response.getCommand().equals("CONNECTED"));
        Assertions.assertTrue((boolean)response.getHeader("response-id").equals("1"));
        this.conn.disconnect();
        try {
            StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "Hello World!");
            Assertions.fail((String)"Should have thrown an exception since connection is disconnected");
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    @TestTemplate
    public void testSendMessage() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "Hello World");
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertEquals((Object)"Hello World", (Object)message.getText());
        Assertions.assertEquals((int)4, (int)message.getJMSPriority(), (String)"getJMSPriority");
        long tnow = System.currentTimeMillis();
        long tmsg = message.getJMSTimestamp();
        Assertions.assertTrue((Math.abs(tnow - tmsg) < 1000L ? (byte)1 : 0) != 0);
    }

    @TestTemplate
    public void testNullCorrelationIDandTypeProperties() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "Hello World");
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertFalse((boolean)((ActiveMQMessage)message).getCoreMessage().getPropertyNames().contains(MessageUtil.CORRELATIONID_HEADER_NAME));
        Assertions.assertFalse((boolean)((ActiveMQMessage)message).getCoreMessage().getPropertyNames().contains(MessageUtil.TYPE_HEADER_NAME));
    }

    @TestTemplate
    public void sendSTOMPReceiveMQTT() throws Exception {
        FuseMQTTClientProvider clientProvider = new FuseMQTTClientProvider();
        clientProvider.connect("tcp://" + this.hostname + ":61613");
        clientProvider.subscribe(StompTest.getTopicPrefix() + StompTest.getTopicName(), 0);
        String stompPayload = "This is a test message";
        this.conn.connect(this.defUser, this.defPass);
        StompTest.send(this.conn, StompTest.getTopicPrefix() + StompTest.getTopicName(), null, stompPayload);
        byte[] mqttPayload = clientProvider.receive(10000);
        clientProvider.disconnect();
        Assertions.assertEquals((Object)stompPayload, (Object)new String(mqttPayload, StandardCharsets.UTF_8));
        clientProvider.disconnect();
    }

    @TestTemplate
    public void sendMQTTReceiveSTOMP() throws Exception {
        String payload = "This is a test message";
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto");
        FuseMQTTClientProvider clientProvider = new FuseMQTTClientProvider();
        clientProvider.connect("tcp://" + this.hostname + ":61613");
        clientProvider.publish(StompTest.getQueuePrefix() + StompTest.getQueueName(), payload.getBytes(), 0);
        clientProvider.disconnect();
        ClientStompFrame frame = this.conn.receiveFrame();
        Assertions.assertTrue((boolean)frame.getBody().contains(payload));
    }

    @TestTemplate
    public void sendEmptyCoreMessage() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto");
        MessageProducer mp = this.session.createProducer((Destination)this.session.createQueue(StompTest.getQueuePrefix() + StompTest.getQueueName()));
        Message m = this.session.createMessage();
        mp.send(m);
        ClientStompFrame frame = this.conn.receiveFrame();
        Assertions.assertNotNull((Object)frame);
        Assertions.assertNull((Object)frame.getBody());
    }

    public void sendMessageToNonExistentQueue(String queuePrefix, String queue, RoutingType routingType) throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.send(this.conn, queuePrefix + queue, null, "Hello World", true, routingType);
        MessageConsumer consumer = this.session.createConsumer((Destination)ActiveMQJMSClient.createQueue((String)queue));
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertEquals((Object)"Hello World", (Object)message.getText());
        Assertions.assertEquals((int)4, (int)message.getJMSPriority(), (String)"getJMSPriority");
        long tnow = System.currentTimeMillis();
        long tmsg = message.getJMSTimestamp();
        Assertions.assertTrue((Math.abs(tnow - tmsg) < 1500L ? (byte)1 : 0) != 0);
        Assertions.assertNotNull((Object)this.server.getPostOffice().getBinding(SimpleString.of((String)queue)));
        consumer.close();
        Wait.assertTrue(() -> this.server.getPostOffice().getBinding(SimpleString.of((String)queue)) == null);
    }

    @TestTemplate
    public void testSendMessageToNonExistentQueue() throws Exception {
        this.sendMessageToNonExistentQueue(StompTest.getQueuePrefix(), RandomUtil.randomString(), RoutingType.ANYCAST);
    }

    @TestTemplate
    public void testSendMessageToNonExistentQueueUsingExplicitDefaultRouting() throws Exception {
        String nonExistentQueue = RandomUtil.randomString();
        this.server.getAddressSettingsRepository().addMatch(nonExistentQueue, (Object)new AddressSettings().setDefaultAddressRoutingType(RoutingType.ANYCAST).setDefaultQueueRoutingType(RoutingType.ANYCAST));
        this.sendMessageToNonExistentQueue(StompTest.getQueuePrefix(), nonExistentQueue, null);
    }

    public void sendMessageToNonExistentTopic(String topicPrefix, String topic, RoutingType routingType) throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.send(this.conn, topicPrefix + topic, null, "Hello World", true, routingType);
        MessageConsumer consumer = this.session.createConsumer((Destination)ActiveMQJMSClient.createTopic((String)topic));
        StompTest.send(this.conn, topicPrefix + topic, null, "Hello World", true, routingType);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertEquals((Object)"Hello World", (Object)message.getText());
        Assertions.assertEquals((int)4, (int)message.getJMSPriority(), (String)"getJMSPriority");
        long tnow = System.currentTimeMillis();
        long tmsg = message.getJMSTimestamp();
        Assertions.assertTrue((Math.abs(tnow - tmsg) < 1500L ? (byte)1 : 0) != 0);
        Assertions.assertNotNull((Object)this.server.getAddressInfo(SimpleString.of((String)topic)));
        consumer.close();
        Wait.assertTrue(() -> this.server.getAddressInfo(SimpleString.of((String)topic)) == null);
    }

    @TestTemplate
    public void testSendMessageToNonExistentTopic() throws Exception {
        this.sendMessageToNonExistentTopic(StompTest.getTopicPrefix(), RandomUtil.randomString(), RoutingType.MULTICAST);
    }

    @TestTemplate
    public void testSendMessageToNonExistentTopicUsingExplicitDefaultRouting() throws Exception {
        String nonExistentTopic = RandomUtil.randomString();
        this.server.getAddressSettingsRepository().addMatch(nonExistentTopic, (Object)new AddressSettings().setDefaultAddressRoutingType(RoutingType.MULTICAST).setDefaultQueueRoutingType(RoutingType.MULTICAST));
        this.sendMessageToNonExistentTopic(StompTest.getTopicPrefix(), nonExistentTopic, null);
    }

    @TestTemplate
    public void testSendMessageToNonExistentTopicUsingImplicitDefaultRouting() throws Exception {
        this.sendMessageToNonExistentTopic(StompTest.getTopicPrefix(), RandomUtil.randomString(), null);
    }

    @TestTemplate
    public void testSendMessageWithLeadingNewLine() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompTest.getQueuePrefix() + StompTest.getQueueName()).setBody("Hello World");
        this.conn.sendWickedFrame(frame);
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertEquals((Object)"Hello World", (Object)message.getText());
        StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "Hello World");
        message = (TextMessage)consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertEquals((Object)"Hello World", (Object)message.getText());
        long tnow = System.currentTimeMillis();
        long tmsg = message.getJMSTimestamp();
        Assertions.assertTrue((Math.abs(tnow - tmsg) < 1000L ? (byte)1 : 0) != 0);
    }

    @TestTemplate
    public void testSendMessageWithReceipt() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "Hello World", true);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertEquals((Object)"Hello World", (Object)message.getText());
        long tnow = System.currentTimeMillis();
        long tmsg = message.getJMSTimestamp();
        Assertions.assertTrue((Math.abs(tnow - tmsg) < 1000L ? (byte)1 : 0) != 0);
    }

    @TestTemplate
    public void testSendMessageWithContentLength() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        byte[] data = new byte[]{1, 0, 0, 4};
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        baos.write(data);
        baos.flush();
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader("content-length", Integer.toString(data.length)).setBody(new String(baos.toByteArray()));
        this.conn.sendFrame(frame);
        BytesMessage message = (BytesMessage)consumer.receive(10000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertEquals((long)data.length, (long)message.getBodyLength());
        Assertions.assertEquals((byte)data[0], (byte)message.readByte());
        Assertions.assertEquals((byte)data[1], (byte)message.readByte());
        Assertions.assertEquals((byte)data[2], (byte)message.readByte());
        Assertions.assertEquals((byte)data[3], (byte)message.readByte());
    }

    @TestTemplate
    public void testJMSXGroupIdCanBeSet() throws Exception {
        String jmsxGroupID = "JMSXGroupID";
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader("JMSXGroupID", "JMSXGroupID").setBody("Hello World");
        this.conn.sendFrame(frame);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertEquals((Object)"Hello World", (Object)message.getText());
        Assertions.assertEquals((Object)"JMSXGroupID", (Object)message.getStringProperty("JMSXGroupID"));
    }

    @TestTemplate
    public void testSendMessageWithCustomHeadersAndSelector() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue, "foo = 'abc'");
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader("foo", "abc").addHeader("bar", "123").setBody("Hello World");
        this.conn.sendFrame(frame);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertEquals((Object)"Hello World", (Object)message.getText());
        Assertions.assertEquals((Object)"abc", (Object)message.getStringProperty("foo"), (String)"foo");
        Assertions.assertEquals((Object)"123", (Object)message.getStringProperty("bar"), (String)"bar");
    }

    @TestTemplate
    public void testSendMessageWithCustomHeadersAndHyphenatedSelector() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue, "hyphenated_props:b-ar = '123'");
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader("foo", "abc").addHeader("b-ar", "123").setBody("Hello World");
        this.conn.sendFrame(frame);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertEquals((Object)"Hello World", (Object)message.getText());
        Assertions.assertEquals((Object)"abc", (Object)message.getStringProperty("foo"), (String)"foo");
        Assertions.assertEquals((Object)"123", (Object)message.getStringProperty("b-ar"), (String)"b-ar");
    }

    @TestTemplate
    public void testSendMessageWithStandardHeaders() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader("foo", "abc").addHeader("bar", "123").addHeader("correlation-id", "c123").addHeader("persistent", "true").addHeader("type", "t345").addHeader("JMSXGroupID", "abc").addHeader("priority", "3").setBody("Hello World");
        this.conn.sendFrame(frame);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertEquals((Object)"Hello World", (Object)message.getText());
        Assertions.assertEquals((Object)"c123", (Object)message.getJMSCorrelationID(), (String)"JMSCorrelationID");
        Assertions.assertEquals((Object)"t345", (Object)message.getJMSType(), (String)"getJMSType");
        Assertions.assertEquals((int)3, (int)message.getJMSPriority(), (String)"getJMSPriority");
        Assertions.assertEquals((int)2, (int)message.getJMSDeliveryMode());
        Assertions.assertEquals((Object)"abc", (Object)message.getStringProperty("foo"), (String)"foo");
        Assertions.assertEquals((Object)"123", (Object)message.getStringProperty("bar"), (String)"bar");
        Assertions.assertEquals((Object)"abc", (Object)message.getStringProperty("JMSXGroupID"), (String)"JMSXGroupID");
    }

    @TestTemplate
    public void testSendMessageWithLongHeaders() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        StringBuffer buffer = new StringBuffer();
        for (int i = 0; i < 1024; ++i) {
            buffer.append("a");
        }
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader("foo", "abc").addHeader("bar", "123").addHeader("correlation-id", "c123").addHeader("persistent", "true").addHeader("type", "t345").addHeader("JMSXGroupID", "abc").addHeader("priority", "3").addHeader("longHeader", buffer.toString()).setBody("Hello World");
        this.conn.sendFrame(frame);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertEquals((Object)"Hello World", (Object)message.getText());
        Assertions.assertEquals((Object)"c123", (Object)message.getJMSCorrelationID(), (String)"JMSCorrelationID");
        Assertions.assertEquals((Object)"t345", (Object)message.getJMSType(), (String)"getJMSType");
        Assertions.assertEquals((int)3, (int)message.getJMSPriority(), (String)"getJMSPriority");
        Assertions.assertEquals((int)2, (int)message.getJMSDeliveryMode());
        Assertions.assertEquals((Object)"abc", (Object)message.getStringProperty("foo"), (String)"foo");
        Assertions.assertEquals((int)1024, (int)message.getStringProperty("longHeader").length(), (String)"longHeader");
        Assertions.assertEquals((Object)"abc", (Object)message.getStringProperty("JMSXGroupID"), (String)"JMSXGroupID");
    }

    @TestTemplate
    public void testSendMessageWithDelay() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader("foo", "abc").addHeader("bar", "123").addHeader("correlation-id", "c123").addHeader("persistent", "true").addHeader("type", "t345").addHeader("JMSXGroupID", "abc").addHeader("priority", "3").addHeader("AMQ_SCHEDULED_DELAY", "2000").setBody("Hello World");
        this.conn.sendFrame(frame);
        Assertions.assertNull((Object)consumer.receive(100L), (String)"Should not receive message yet");
        TextMessage message = (TextMessage)consumer.receive(4000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertEquals((Object)"Hello World", (Object)message.getText());
        Assertions.assertEquals((Object)"c123", (Object)message.getJMSCorrelationID(), (String)"JMSCorrelationID");
        Assertions.assertEquals((Object)"t345", (Object)message.getJMSType(), (String)"getJMSType");
        Assertions.assertEquals((int)3, (int)message.getJMSPriority(), (String)"getJMSPriority");
        Assertions.assertEquals((int)2, (int)message.getJMSDeliveryMode());
        Assertions.assertEquals((Object)"abc", (Object)message.getStringProperty("foo"), (String)"foo");
        Assertions.assertEquals((Object)"abc", (Object)message.getStringProperty("JMSXGroupID"), (String)"JMSXGroupID");
    }

    @TestTemplate
    public void testSendMessageWithDeliveryTime() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader("foo", "abc").addHeader("bar", "123").addHeader("correlation-id", "c123").addHeader("persistent", "true").addHeader("type", "t345").addHeader("JMSXGroupID", "abc").addHeader("priority", "3").addHeader("AMQ_SCHEDULED_TIME", Long.toString(System.currentTimeMillis() + 2000L)).setBody("Hello World");
        this.conn.sendFrame(frame);
        Assertions.assertNull((Object)consumer.receive(100L), (String)"Should not receive message yet");
        TextMessage message = (TextMessage)consumer.receive(4000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertEquals((Object)"Hello World", (Object)message.getText());
        Assertions.assertEquals((Object)"c123", (Object)message.getJMSCorrelationID(), (String)"JMSCorrelationID");
        Assertions.assertEquals((Object)"t345", (Object)message.getJMSType(), (String)"getJMSType");
        Assertions.assertEquals((int)3, (int)message.getJMSPriority(), (String)"getJMSPriority");
        Assertions.assertEquals((int)2, (int)message.getJMSDeliveryMode());
        Assertions.assertEquals((Object)"abc", (Object)message.getStringProperty("foo"), (String)"foo");
        Assertions.assertEquals((Object)"abc", (Object)message.getStringProperty("JMSXGroupID"), (String)"JMSXGroupID");
    }

    @TestTemplate
    public void testSendMessageWithDelayWithBadValue() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader("AMQ_SCHEDULED_DELAY", "foo").setBody("Hello World");
        this.conn.sendFrame(frame);
        Assertions.assertNull((Object)consumer.receive(100L), (String)"Should not receive message yet");
        ClientStompFrame error = this.conn.receiveFrame();
        Assertions.assertNotNull((Object)error);
        Assertions.assertTrue((boolean)error.getCommand().equals("ERROR"));
    }

    @TestTemplate
    public void testSendMessageWithDeliveryTimeWithBadValue() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader("AMQ_SCHEDULED_TIME", "foo").setBody("Hello World");
        this.conn.sendFrame(frame);
        Assertions.assertNull((Object)consumer.receive(100L), (String)"Should not receive message yet");
        ClientStompFrame error = this.conn.receiveFrame();
        Assertions.assertNotNull((Object)error);
        Assertions.assertTrue((boolean)error.getCommand().equals("ERROR"));
    }

    @TestTemplate
    public void testSubscribeWithAutoAck() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)(StompTest.getQueuePrefix() + StompTest.getQueueName()), (Object)frame.getHeader("destination"));
        Assertions.assertEquals((Object)this.getName(), (Object)frame.getBody());
        Pattern cl = Pattern.compile("content-length:\\s*(\\d+)", 2);
        Matcher cl_matcher = cl.matcher(frame.toString());
        Assertions.assertFalse((boolean)cl_matcher.find());
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(100L);
        Assertions.assertNull((Object)message);
    }

    @TestTemplate
    public void testNullPropertyValue() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto");
        this.sendJmsMessage(this.getName(), "foo", null);
        ClientStompFrame frame = this.conn.receiveFrame(2000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)(StompTest.getQueuePrefix() + StompTest.getQueueName()), (Object)frame.getHeader("destination"));
        Assertions.assertEquals((Object)this.getName(), (Object)frame.getBody());
        Assertions.assertEquals((Object)"", (Object)frame.getHeader("foo"));
        this.conn.disconnect();
    }

    @TestTemplate
    public void testTransactedSessionLeak() throws Exception {
        for (int i = 0; i < 10; ++i) {
            this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
            this.conn.connect(this.defUser, this.defPass);
            for (int s = 0; s < 10; ++s) {
                String txId = "tx" + i + "_" + s;
                StompTest.beginTransaction(this.conn, txId);
                StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "Hello World", true, null, txId);
                StompTest.commitTransaction(this.conn, txId, true);
            }
            Wait.assertEquals((int)13, () -> this.server.getSessions().size(), (long)1000L, (long)100L);
            this.conn.disconnect();
        }
        if (this.connection != null) {
            this.connection.close();
        }
        Wait.assertEquals((int)0, () -> this.server.getSessions().size(), (long)1000L, (long)100L);
        Acceptor stompAcceptor = (Acceptor)this.server.getRemotingService().getAcceptors().get("stomp");
        StompProtocolManager stompProtocolManager = (StompProtocolManager)stompAcceptor.getProtocolHandler().getProtocolMap().get("STOMP");
        Assertions.assertNotNull((Object)stompProtocolManager);
        Assertions.assertEquals((int)0, (int)stompProtocolManager.getTransactedSessions().size());
    }

    @TestTemplate
    public void testIngressTimestamp() throws Exception {
        this.server.getAddressSettingsRepository().addMatch("#", (Object)new AddressSettings().setEnableIngressTimestamp(true));
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto");
        long beforeSend = System.currentTimeMillis();
        this.sendJmsMessage(this.getName());
        long afterSend = System.currentTimeMillis();
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)(StompTest.getQueuePrefix() + StompTest.getQueueName()), (Object)frame.getHeader("destination"));
        String ingressTimestampHeader = frame.getHeader("ingress-timestamp");
        Assertions.assertNotNull((Object)ingressTimestampHeader);
        long ingressTimestamp = Long.parseLong(ingressTimestampHeader);
        Assertions.assertTrue((ingressTimestamp >= beforeSend && ingressTimestamp <= afterSend ? (byte)1 : 0) != 0, (String)("Ingress timstamp " + ingressTimestamp + " should be >= " + beforeSend + " and <= " + afterSend));
        this.conn.disconnect();
    }

    @TestTemplate
    public void testAnycastDestinationTypeMessageProperty() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto");
        StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, this.getName(), true, RoutingType.ANYCAST);
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)(StompTest.getQueuePrefix() + StompTest.getQueueName()), (Object)frame.getHeader("destination"));
        Assertions.assertEquals((Object)RoutingType.ANYCAST.toString(), (Object)frame.getHeader("destination-type"));
        Assertions.assertTrue((frame.getHeader(org.apache.activemq.artemis.api.core.Message.HDR_ROUTING_TYPE.toString()) == null ? (byte)1 : 0) != 0);
        Assertions.assertEquals((Object)this.getName(), (Object)frame.getBody());
        this.conn.disconnect();
    }

    @TestTemplate
    public void testMulticastDestinationTypeMessageProperty() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribeTopic(this.conn, null, null, null);
        StompTest.send(this.conn, StompTest.getTopicPrefix() + StompTest.getTopicName(), null, this.getName(), true, RoutingType.MULTICAST);
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)(StompTest.getTopicPrefix() + StompTest.getTopicName()), (Object)frame.getHeader("destination"));
        Assertions.assertEquals((Object)RoutingType.MULTICAST.toString(), (Object)frame.getHeader("destination-type"));
        Assertions.assertTrue((frame.getHeader(org.apache.activemq.artemis.api.core.Message.HDR_ROUTING_TYPE.toString()) == null ? (byte)1 : 0) != 0);
        Assertions.assertEquals((Object)this.getName(), (Object)frame.getBody());
        this.conn.disconnect();
    }

    @TestTemplate
    public void testSubscriptionQueueCreatedWhenAutoCreateDisabled() throws Exception {
        SimpleString topic = SimpleString.of((String)(StompTest.getTopicPrefix() + StompTest.getTopicName()));
        ((AddressSettings)this.server.getAddressSettingsRepository().getMatch(topic.toString())).setAutoCreateQueues(Boolean.valueOf(false));
        this.conn.connect(this.defUser, this.defPass);
        Assertions.assertEquals((int)0, (int)this.server.getPostOffice().getBindingsForAddress(topic).size());
        StompTest.subscribeTopic(this.conn, null, null, null, true);
        Wait.assertEquals((int)1, () -> this.server.getPostOffice().getBindingsForAddress(topic).size(), (long)2000L, (long)100L);
        this.conn.disconnect();
    }

    @TestTemplate
    public void testSubscribeWithAutoAckAndBytesMessage() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto");
        byte[] payload = new byte[]{1, 2, 3, 4, 5};
        this.sendJmsMessage(payload, (Destination)this.queue);
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Pattern cl = Pattern.compile("content-length:\\s*(\\d+)", 2);
        Matcher cl_matcher = cl.matcher(frame.toString());
        Assertions.assertTrue((boolean)cl_matcher.find());
        Assertions.assertEquals((Object)"5", (Object)cl_matcher.group(1));
        Assertions.assertFalse((boolean)Pattern.compile("type:\\s*null", 2).matcher(frame.toString()).find());
        Assertions.assertTrue((frame.getBody().toString().indexOf(new String(payload)) > -1 ? (byte)1 : 0) != 0);
        this.conn.disconnect();
    }

    @TestTemplate
    public void testSubscribeWithMessageSentWithProperties() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto");
        MessageProducer producer = this.session.createProducer((Destination)this.queue);
        BytesMessage message = this.session.createBytesMessage();
        message.setStringProperty("S", "value");
        message.setBooleanProperty("n", false);
        message.setByteProperty("byte", (byte)9);
        message.setDoubleProperty("d", 2.0);
        message.setFloatProperty("f", 6.0f);
        message.setIntProperty("i", 10);
        message.setLongProperty("l", 121L);
        message.setShortProperty("s", (short)12);
        message.writeBytes("Hello World".getBytes(StandardCharsets.UTF_8));
        producer.send((Message)message);
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assertions.assertNotNull((Object)frame);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)"value", (Object)frame.getHeader("S"));
        Assertions.assertEquals((Object)"false", (Object)frame.getHeader("n"));
        Assertions.assertEquals((Object)"9", (Object)frame.getHeader("byte"));
        Assertions.assertEquals((Object)"2.0", (Object)frame.getHeader("d"));
        Assertions.assertEquals((Object)"6.0", (Object)frame.getHeader("f"));
        Assertions.assertEquals((Object)"10", (Object)frame.getHeader("i"));
        Assertions.assertEquals((Object)"121", (Object)frame.getHeader("l"));
        Assertions.assertEquals((Object)"12", (Object)frame.getHeader("s"));
        Assertions.assertEquals((Object)"Hello World", (Object)frame.getBody());
        this.conn.disconnect();
    }

    @TestTemplate
    public void testSubscribeWithID() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, "mysubid", "auto");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)(StompTest.getQueuePrefix() + StompTest.getQueueName()), (Object)frame.getHeader("destination"));
        Assertions.assertEquals((Object)"mysubid", (Object)frame.getHeader("subscription"));
        Assertions.assertEquals((Object)this.getName(), (Object)frame.getBody());
        this.conn.disconnect();
    }

    @TestTemplate
    public void testBodyWithUTF8() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto");
        String text = "A\u00ea\u00f1\u00fcC";
        logger.debug(text);
        this.sendJmsMessage(text);
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        logger.debug("{}", (Object)frame);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)(StompTest.getQueuePrefix() + StompTest.getQueueName()), (Object)frame.getHeader("destination"));
        Assertions.assertEquals((Object)text, (Object)frame.getBody());
        this.conn.disconnect();
    }

    @TestTemplate
    public void testMessagesAreInOrder() throws Exception {
        ClientStompFrame frame;
        int i;
        int ctr = 10;
        String[] data = new String[ctr];
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto");
        for (i = 0; i < ctr; ++i) {
            data[i] = this.getName() + i;
            this.sendJmsMessage(data[i]);
        }
        for (i = 0; i < ctr; ++i) {
            frame = this.conn.receiveFrame(1000L);
            Assertions.assertTrue((boolean)frame.getBody().equals(data[i]), (String)"Message not in order");
        }
        Thread.sleep(200L);
        for (i = 0; i < ctr; ++i) {
            data[i] = this.getName() + ":second:" + i;
            this.sendJmsMessage(data[i]);
        }
        for (i = 0; i < ctr; ++i) {
            frame = this.conn.receiveFrame(1000L);
            Assertions.assertTrue((boolean)frame.getBody().equals(data[i]), (String)"Message not in order");
        }
        this.conn.disconnect();
    }

    @TestTemplate
    public void testSubscribeWithAutoAckAndSelector() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto", null, "foo = 'zzz'");
        this.sendJmsMessage("Ignored message", "foo", "1234");
        this.sendJmsMessage("Real message", "foo", "zzz");
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertTrue((boolean)frame.getBody().equals("Real message"), (String)("Should have received the real message but got: " + frame));
        this.conn.disconnect();
    }

    @TestTemplate
    public void testSubscribeWithAutoAckAndHyphenatedSelector() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto", null, "hyphenated_props:foo-bar = 'zzz'");
        ServerLocator serverLocator = this.addServerLocator(ActiveMQClient.createServerLocator((String)"vm://0"));
        ClientSessionFactory clientSessionFactory = serverLocator.createSessionFactory();
        ClientSession clientSession = clientSessionFactory.createSession(true, true);
        ClientProducer producer = clientSession.createProducer(StompTest.getQueuePrefix() + StompTest.getQueueName());
        ClientMessage ignoredMessage = clientSession.createMessage(false);
        ignoredMessage.putStringProperty("foo-bar", "1234");
        ignoredMessage.getBodyBuffer().writeNullableSimpleString(SimpleString.of((String)"Ignored message"));
        ClientMessage realMessage = clientSession.createMessage(false);
        realMessage.putStringProperty("foo-bar", "zzz");
        realMessage.getBodyBuffer().writeNullableSimpleString(SimpleString.of((String)"Real message"));
        producer.send((org.apache.activemq.artemis.api.core.Message)ignoredMessage);
        producer.send((org.apache.activemq.artemis.api.core.Message)realMessage);
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertTrue((boolean)frame.getBody().equals("Real message"), (String)("Should have received the real message but got: " + frame));
        this.conn.disconnect();
    }

    @TestTemplate
    public void testSubscribeWithAutoAckAndXpathSelector() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto", null, "XPATH 'root/a'");
        this.sendJmsMessage("<root><b key='first' num='1'/><c key='second' num='2'>c</c></root>");
        this.sendJmsMessage("<root><a key='first' num='1'/><b key='second' num='2'>b</b></root>");
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertTrue((boolean)frame.getBody().equals("<root><a key='first' num='1'/><b key='second' num='2'>b</b></root>"), (String)("Should have received the real message but got: " + frame));
        this.conn.disconnect();
    }

    @TestTemplate
    public void testSubscribeWithClientAck() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertNotNull((Object)frame.getHeader("message-id"));
        StompTest.ack(this.conn, null, frame);
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(100L);
        Assertions.assertNull((Object)message);
    }

    @TestTemplate
    public void testRedeliveryWithClientAck() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        this.conn.disconnect();
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        Message message = consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertTrue((boolean)message.getJMSRedelivered());
    }

    @TestTemplate
    public void testSubscribeWithClientAckThenConsumingAgainWithAutoAckWithNoDisconnectFrame() throws Exception {
        this.assertSubscribeWithClientAckThenConsumeWithAutoAck(false);
    }

    @TestTemplate
    public void testSubscribeWithClientAckThenConsumingAgainWithAutoAckWithExplicitDisconnect() throws Exception {
        this.assertSubscribeWithClientAckThenConsumeWithAutoAck(true);
    }

    protected void assertSubscribeWithClientAckThenConsumeWithAutoAck(boolean sendDisconnect) throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        logger.debug("Reconnecting!");
        if (sendDisconnect) {
            this.conn.disconnect();
            this.conn.destroy();
            this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        } else {
            this.conn.destroy();
            this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        }
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null);
        frame = this.conn.receiveFrame(10000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        this.conn.disconnect();
        this.conn.destroy();
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto", null, true);
        this.sendJmsMessage("shouldBeNextMessage");
        frame = this.conn.receiveFrame(10000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)"shouldBeNextMessage", (Object)frame.getBody());
    }

    @TestTemplate
    public void testUnsubscribe() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto");
        this.sendJmsMessage("first message");
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        StompTest.unsubscribe(this.conn, null, StompTest.getQueuePrefix() + StompTest.getQueueName(), true, false);
        this.sendJmsMessage("second message");
        frame = this.conn.receiveFrame(100L);
        logger.debug("Received frame: {}", (Object)frame);
        Assertions.assertNull((Object)frame, (String)"No message should have been received since subscription was removed");
    }

    @TestTemplate
    public void testUnsubscribeWithID() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, "mysubid", "auto");
        this.sendJmsMessage("first message");
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        StompTest.unsubscribe(this.conn, "mysubid", null, true, false);
        this.sendJmsMessage("second message");
        frame = this.conn.receiveFrame(100L);
        logger.debug("Received frame: {}", (Object)frame);
        Assertions.assertNull((Object)frame, (String)"No message should have been received since subscription was removed");
    }

    @TestTemplate
    public void testTransactionCommit() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        StompTest.beginTransaction(this.conn, "tx1");
        StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "Hello World", true, null, "tx1");
        Assertions.assertNull((Object)consumer.receive(100L));
        StompTest.commitTransaction(this.conn, "tx1", true);
        Message message = consumer.receive(1000L);
        Assertions.assertNotNull((Object)message, (String)"Should have received a message");
    }

    @TestTemplate
    public void testSuccessiveTransactionsWithSameID() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        this.conn.connect(this.defUser, this.defPass);
        StompTest.beginTransaction(this.conn, "tx1");
        StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "Hello World", true, null, "tx1");
        StompTest.commitTransaction(this.conn, "tx1");
        Message message = consumer.receive(1000L);
        Assertions.assertNotNull((Object)message, (String)"Should have received a message");
        StompTest.beginTransaction(this.conn, "tx1");
        StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "Hello World", true, null, "tx1");
        StompTest.commitTransaction(this.conn, "tx1");
        message = consumer.receive(1000L);
        Assertions.assertNotNull((Object)message, (String)"Should have received a message");
    }

    @TestTemplate
    public void testBeginSameTransactionTwice() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.beginTransaction(this.conn, "tx1");
        StompTest.beginTransaction(this.conn, "tx1");
        ClientStompFrame frame = this.conn.receiveFrame(1000L);
        Assertions.assertEquals((Object)"ERROR", (Object)frame.getCommand());
    }

    @TestTemplate
    public void testTransactionRollback() throws Exception {
        MessageConsumer consumer = this.session.createConsumer((Destination)this.queue);
        String txId = "tx1";
        this.conn.connect(this.defUser, this.defPass);
        StompTest.beginTransaction(this.conn, txId);
        StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "first message", true, null, txId);
        StompTest.abortTransaction(this.conn, txId);
        StompTest.beginTransaction(this.conn, txId);
        StompTest.send(this.conn, StompTest.getQueuePrefix() + StompTest.getQueueName(), null, "second message", true, null, txId);
        StompTest.commitTransaction(this.conn, txId);
        TextMessage message = (TextMessage)consumer.receive(1000L);
        Assertions.assertNotNull((Object)message);
        Assertions.assertEquals((Object)"second message", (Object)message.getText());
    }

    @TestTemplate
    public void testSubscribeToTopic() throws Exception {
        int baselineQueueCount = this.server.getActiveMQServerControl().getQueueNames().length;
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribeTopic(this.conn, null, null, null, true);
        Wait.assertTrue((String)"Subscription queue should be created here", () -> {
            int length = this.server.getActiveMQServerControl().getQueueNames().length;
            if (length - baselineQueueCount == 1) {
                return true;
            }
            logger.debug("Queue count: {}", (Object)(length - baselineQueueCount));
            return false;
        });
        this.sendJmsMessage(this.getName(), (Destination)this.topic);
        ClientStompFrame frame = this.conn.receiveFrame(1000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)(StompTest.getTopicPrefix() + StompTest.getTopicName()), (Object)frame.getHeader("destination"));
        Assertions.assertEquals((Object)this.getName(), (Object)frame.getBody());
        StompTest.unsubscribe(this.conn, null, StompTest.getTopicPrefix() + StompTest.getTopicName(), true, false);
        this.sendJmsMessage(this.getName(), (Destination)this.topic);
        frame = this.conn.receiveFrame(100L);
        logger.debug("Received frame: {}", (Object)frame);
        Assertions.assertNull((Object)frame, (String)"No message should have been received since subscription was removed");
        Assertions.assertEquals((int)0, (int)(this.server.getActiveMQServerControl().getQueueNames().length - baselineQueueCount), (String)"Subscription queue should be deleted");
        this.conn.disconnect();
    }

    @TestTemplate
    public void testSubscribeToQueue() throws Exception {
        int baselineQueueCount = this.server.getActiveMQServerControl().getQueueNames().length;
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, null, null, true);
        Wait.assertFalse((String)"Queue should not be created here", () -> this.server.getActiveMQServerControl().getQueueNames().length - baselineQueueCount == 1);
        this.sendJmsMessage(this.getName(), (Destination)this.queue);
        ClientStompFrame frame = this.conn.receiveFrame(1000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)(StompTest.getQueuePrefix() + StompTest.getQueueName()), (Object)frame.getHeader("destination"));
        Assertions.assertEquals((Object)this.getName(), (Object)frame.getBody());
        StompTest.unsubscribe(this.conn, null, StompTest.getQueuePrefix() + StompTest.getQueueName(), true, false);
        this.sendJmsMessage(this.getName(), (Destination)this.queue);
        frame = this.conn.receiveFrame(100L);
        logger.debug("Received frame: {}", (Object)frame);
        Assertions.assertNull((Object)frame, (String)"No message should have been received since subscription was removed");
        Assertions.assertEquals((int)baselineQueueCount, (int)this.server.getActiveMQServerControl().getQueueNames().length, (String)"Subscription queue should not be deleted");
        this.conn.disconnect();
    }

    @TestTemplate
    public void testSubscribeToNonExistentQueue() throws Exception {
        String nonExistentQueue = RandomUtil.randomString();
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, null, null, null, StompTest.getQueuePrefix() + nonExistentQueue, true);
        this.sendJmsMessage(this.getName(), (Destination)ActiveMQJMSClient.createQueue((String)nonExistentQueue));
        ClientStompFrame frame = this.conn.receiveFrame(1000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)(StompTest.getQueuePrefix() + nonExistentQueue), (Object)frame.getHeader("destination"));
        Assertions.assertEquals((Object)this.getName(), (Object)frame.getBody());
        Assertions.assertNotNull((Object)this.server.getPostOffice().getBinding(SimpleString.of((String)nonExistentQueue)));
        Queue subscription = ((LocalQueueBinding)this.server.getPostOffice().getBinding(SimpleString.of((String)nonExistentQueue))).getQueue();
        Wait.assertEquals((long)0L, () -> ((Queue)subscription).getMessageCount());
        StompTest.unsubscribe(this.conn, null, StompTest.getQueuePrefix() + nonExistentQueue, true, false);
        Wait.assertTrue(() -> this.server.getPostOffice().getBinding(SimpleString.of((String)nonExistentQueue)) == null);
        this.sendJmsMessage(this.getName(), (Destination)ActiveMQJMSClient.createQueue((String)nonExistentQueue));
        frame = this.conn.receiveFrame(100L);
        logger.debug("Received frame: {}", (Object)frame);
        Assertions.assertNull((Object)frame, (String)"No message should have been received since subscription was removed");
        this.conn.disconnect();
    }

    @TestTemplate
    public void testDurableSubscriberWithReconnection() throws Exception {
        this.conn.connect(this.defUser, this.defPass, "myclientid");
        StompTest.subscribeTopic(this.conn, null, null, this.getName());
        this.conn.disconnect();
        Thread.sleep(500L);
        this.sendJmsMessage(this.getName(), (Destination)this.topic);
        this.conn.destroy();
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        this.conn.connect(this.defUser, this.defPass, "myclientid");
        StompTest.subscribeTopic(this.conn, null, null, this.getName());
        ClientStompFrame frame = this.conn.receiveFrame(3000L);
        Assertions.assertNotNull((Object)frame, (String)"Should have received a message from the durable subscription");
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)(StompTest.getTopicPrefix() + StompTest.getTopicName()), (Object)frame.getHeader("destination"));
        Assertions.assertEquals((Object)this.getName(), (Object)frame.getBody());
        StompTest.unsubscribe(this.conn, null, StompTest.getTopicPrefix() + StompTest.getTopicName(), true, true);
        this.conn.disconnect();
    }

    @TestTemplate
    public void testDurableSubscriberWithReconnectionLegacy() throws Exception {
        this.conn.connect(this.defUser, this.defPass, "myclientid");
        StompTest.subscribeTopicLegacyActiveMQ(this.conn, null, null, this.getName(), true, false);
        this.conn.disconnect();
        Thread.sleep(500L);
        this.sendJmsMessage(this.getName(), (Destination)this.topic);
        this.conn.destroy();
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        this.conn.connect(this.defUser, this.defPass, "myclientid");
        StompTest.subscribeTopicLegacyActiveMQ(this.conn, null, null, this.getName(), true, false);
        ClientStompFrame frame = this.conn.receiveFrame(3000L);
        Assertions.assertNotNull((Object)frame, (String)"Should have received a message from the durable subscription");
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)(StompTest.getTopicPrefix() + StompTest.getTopicName()), (Object)frame.getHeader("destination"));
        Assertions.assertEquals((Object)this.getName(), (Object)frame.getBody());
        StompTest.unsubscribeLegacyActiveMQ(this.conn, null, StompTest.getTopicPrefix() + StompTest.getTopicName(), true, true);
        this.conn.disconnect();
    }

    @TestTemplate
    public void testDurableSubscriber() throws Exception {
        this.conn.connect(this.defUser, this.defPass, "myclientid");
        StompTest.subscribeTopic(this.conn, null, null, this.getName(), true);
        ClientStompFrame response = StompTest.subscribeTopic(this.conn, null, null, this.getName(), true);
        Assertions.assertEquals((Object)"ERROR", (Object)response.getCommand());
        this.conn.disconnect();
    }

    @TestTemplate
    public void testDurableSubscriberLegacySubscriptionHeader() throws Exception {
        this.conn.connect(this.defUser, this.defPass, "myclientid");
        StompTest.subscribeTopicLegacyActiveMQ(this.conn, null, null, this.getName(), true, false);
        ClientStompFrame response = StompTest.subscribeTopicLegacyActiveMQ(this.conn, null, null, this.getName(), true, false);
        Assertions.assertEquals((Object)"ERROR", (Object)response.getCommand());
        this.conn.disconnect();
    }

    @TestTemplate
    public void testDurableUnSubscribe() throws Exception {
        this.conn.connect(this.defUser, this.defPass, "myclientid");
        StompTest.subscribeTopic(this.conn, null, null, this.getName(), true);
        this.conn.disconnect();
        Thread.sleep(500L);
        Assertions.assertNotNull((Object)this.server.locateQueue(SimpleString.of((String)("myclientid." + this.getName()))));
        this.conn.destroy();
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        this.conn.connect(this.defUser, this.defPass, "myclientid");
        StompTest.unsubscribe(this.conn, this.getName(), StompTest.getTopicPrefix() + StompTest.getTopicName(), false, true);
        this.conn.disconnect();
        Thread.sleep(500L);
        Wait.assertTrue(() -> this.server.locateQueue(SimpleString.of((String)("myclientid." + this.getName()))) == null);
    }

    @TestTemplate
    public void testDurableUnSubscribeWithoutDurableSubName() throws Exception {
        this.server.getConfiguration().getWildcardConfiguration().setDelimiter('/');
        this.server.getAddressSettingsRepository().addMatch("/topic/#", (Object)new AddressSettings().setDefaultAddressRoutingType(RoutingType.MULTICAST).setDefaultQueueRoutingType(RoutingType.MULTICAST));
        this.conn.connect(this.defUser, this.defPass, "myclientid");
        String subId = UUID.randomUUID().toString();
        String durableSubName = UUID.randomUUID().toString();
        String receipt = UUID.randomUUID().toString();
        ClientStompFrame frame = this.conn.createFrame("SUBSCRIBE").addHeader("destination", "/topic/test.foo").addHeader("id", subId).addHeader("ack", "client-individual").addHeader("durable-subscription-name", durableSubName).addHeader("receipt", receipt);
        frame = this.conn.sendFrame(frame);
        Assertions.assertEquals((Object)receipt, (Object)frame.getHeader("receipt-id"));
        Wait.assertTrue(() -> this.server.locateQueue(SimpleString.of((String)("myclientid." + durableSubName))) != null);
        receipt = UUID.randomUUID().toString();
        frame = this.conn.createFrame("UNSUBSCRIBE").addHeader("id", subId).addHeader("receipt", receipt);
        frame = this.conn.sendFrame(frame);
        Assertions.assertEquals((Object)receipt, (Object)frame.getHeader("receipt-id"));
        this.conn.disconnect();
        Wait.assertTrue(() -> this.server.locateQueue(SimpleString.of((String)("myclientid." + durableSubName))) != null);
    }

    @TestTemplate
    public void testDurableUnSubscribeLegacySubscriptionHeader() throws Exception {
        this.conn.connect(this.defUser, this.defPass, "myclientid");
        StompTest.subscribeTopicLegacyActiveMQ(this.conn, null, null, this.getName(), true, false);
        this.conn.disconnect();
        Thread.sleep(500L);
        Assertions.assertNotNull((Object)this.server.locateQueue(SimpleString.of((String)("myclientid." + this.getName()))));
        this.conn.destroy();
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        this.conn.connect(this.defUser, this.defPass, "myclientid");
        StompTest.unsubscribeLegacyActiveMQ(this.conn, this.getName(), StompTest.getTopicPrefix() + StompTest.getTopicName(), false, true);
        this.conn.disconnect();
        Thread.sleep(500L);
        Wait.assertTrue(() -> this.server.locateQueue(SimpleString.of((String)("myclientid." + this.getName()))) == null);
    }

    @TestTemplate
    public void testSubscribeToTopicWithNoLocal() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribeTopic(this.conn, null, null, null, true, true);
        StompTest.send(this.conn, StompTest.getTopicPrefix() + StompTest.getTopicName(), null, "Hello World");
        ClientStompFrame frame = this.conn.receiveFrame(100L);
        logger.debug("Received frame: {}", (Object)frame);
        Assertions.assertNull((Object)frame, (String)"No message should have been received since subscription was removed");
        this.sendJmsMessage(this.getName(), (Destination)this.topic);
        frame = this.conn.receiveFrame(10000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)(StompTest.getTopicPrefix() + StompTest.getTopicName()), (Object)frame.getHeader("destination"));
        Assertions.assertEquals((Object)this.getName(), (Object)frame.getBody());
        this.conn.disconnect();
    }

    @TestTemplate
    public void testTopicExistsAfterNoUnsubscribeDisconnect() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribeTopic(this.conn, null, null, null, true);
        this.conn.disconnect();
        Thread.sleep(500L);
        this.conn.destroy();
        this.conn = StompClientConnectionFactory.createClientConnection(this.uri);
        this.conn.connect(this.defUser, this.defPass);
        ClientStompFrame response = StompTest.send(this.conn, StompTest.getTopicPrefix() + StompTest.getTopicName(), null, "Hello World", true);
        Assertions.assertEquals((Object)"RECEIPT", (Object)response.getCommand());
        ClientStompFrame frame = this.conn.receiveFrame(100L);
        logger.debug("Received frame: {}", (Object)frame);
        Assertions.assertNull((Object)frame);
        this.conn.disconnect();
    }

    @TestTemplate
    public void testClientAckNotPartOfTransaction() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "client");
        this.sendJmsMessage(this.getName());
        ClientStompFrame frame = this.conn.receiveFrame(10000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)(StompTest.getQueuePrefix() + StompTest.getQueueName()), (Object)frame.getHeader("destination"));
        String messageID = frame.getHeader("message-id");
        Assertions.assertNotNull((Object)messageID);
        Assertions.assertEquals((Object)this.getName(), (Object)frame.getBody());
        StompTest.beginTransaction(this.conn, "tx1");
        StompTest.ack(this.conn, null, messageID, "tx1");
        StompTest.abortTransaction(this.conn, "tx1");
        frame = this.conn.receiveFrame(100L);
        Assertions.assertNull((Object)frame, (String)"No message should have been received as the message was acked even though the transaction has been aborted");
        StompTest.unsubscribe(this.conn, null, StompTest.getQueuePrefix() + StompTest.getQueueName(), false, false);
        this.conn.disconnect();
    }

    @TestTemplate
    public void testMultiProtocolConsumers() throws Exception {
        int TIME_OUT = 2000;
        int NEGATIVE_TIME_OUT = 100;
        int count = 1000;
        MessageConsumer consumer1 = this.session.createConsumer((Destination)this.topic);
        MessageConsumer consumer2 = this.session.createConsumer((Destination)this.topic);
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribeTopic(this.conn, null, null, null, true);
        MessageProducer producer = this.session.createProducer((Destination)this.topic);
        TextMessage message = this.session.createTextMessage(this.getName());
        for (int i = 1; i <= count; ++i) {
            producer.send((Message)message);
            Assertions.assertNotNull((Object)consumer1.receive(2000L));
            Assertions.assertNotNull((Object)consumer2.receive(2000L));
            ClientStompFrame frame = this.conn.receiveFrame(2000L);
            Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
            Assertions.assertEquals((Object)(StompTest.getTopicPrefix() + StompTest.getTopicName()), (Object)frame.getHeader("destination"));
            Assertions.assertEquals((Object)this.getName(), (Object)frame.getBody());
        }
        consumer1.close();
        consumer2.close();
        StompTest.unsubscribe(this.conn, null, StompTest.getTopicPrefix() + StompTest.getTopicName(), true, false);
        this.sendJmsMessage(this.getName(), (Destination)this.topic);
        ClientStompFrame frame = this.conn.receiveFrame(100L);
        logger.debug("Received frame: {}", (Object)frame);
        Assertions.assertNull((Object)frame, (String)"No message should have been received since subscription was removed");
        this.conn.disconnect();
    }

    @TestTemplate
    public void testUnexpectedAck() throws Exception {
        String messageID = "888888";
        this.conn.connect(this.defUser, this.defPass);
        StompTest.ack(this.conn, null, messageID, null);
        ClientStompFrame frame = this.conn.receiveFrame(1000L);
        Assertions.assertNotNull((Object)frame);
        Assertions.assertEquals((Object)"ERROR", (Object)frame.getCommand());
        this.conn.disconnect();
    }

    @TestTemplate
    public void testDotAnycastPrefixOnSend() throws Exception {
        this.testPrefix("jms.queue.", RoutingType.ANYCAST, true);
    }

    @TestTemplate
    public void testDotMulticastPrefixOnSend() throws Exception {
        this.testPrefix("jms.topic.", RoutingType.MULTICAST, true);
    }

    @TestTemplate
    public void testDotAnycastPrefixOnSubscribe() throws Exception {
        this.testPrefix("jms.queue.", RoutingType.ANYCAST, false);
    }

    @TestTemplate
    public void testDotMulticastPrefixOnSubscribe() throws Exception {
        this.testPrefix("jms.topic.", RoutingType.MULTICAST, false);
    }

    @TestTemplate
    public void testSlashAnycastPrefixOnSend() throws Exception {
        this.testPrefix("/queue/", RoutingType.ANYCAST, true);
    }

    @TestTemplate
    public void testSlashMulticastPrefixOnSend() throws Exception {
        this.testPrefix("/topic/", RoutingType.MULTICAST, true);
    }

    @TestTemplate
    public void testSlashAnycastPrefixOnSubscribe() throws Exception {
        this.testPrefix("/queue/", RoutingType.ANYCAST, false);
    }

    @TestTemplate
    public void testSlashMulticastPrefixOnSubscribe() throws Exception {
        this.testPrefix("/topic/", RoutingType.MULTICAST, false);
    }

    public void testPrefix(String prefix, RoutingType routingType, boolean send) throws Exception {
        int port = 61614;
        URI uri = StompTest.createStompClientUri(this.scheme, this.hostname, port);
        String ADDRESS = UUID.randomUUID().toString();
        String PREFIXED_ADDRESS = prefix + ADDRESS;
        String param = routingType.toString();
        String urlParam = param.toLowerCase() + "Prefix";
        this.server.getRemotingService().createAcceptor("test", "tcp://" + this.hostname + ":" + port + "?protocols=STOMP&" + urlParam + "=" + prefix).start();
        StompClientConnection conn = StompClientConnectionFactory.createClientConnection(uri);
        conn.connect(this.defUser, this.defPass);
        if (send) {
            StompTest.send(conn, PREFIXED_ADDRESS, null, "Hello World", true);
        } else {
            String uuid = UUID.randomUUID().toString();
            ClientStompFrame frame = conn.createFrame("SUBSCRIBE").addHeader("destination", PREFIXED_ADDRESS).addHeader("receipt", uuid);
            frame = conn.sendFrame(frame);
            Assertions.assertEquals((Object)uuid, (Object)frame.getHeader("receipt-id"));
        }
        AddressInfo addressInfo = this.server.getAddressInfo(SimpleString.of((String)ADDRESS));
        Assertions.assertNotNull((Object)addressInfo, (String)("No address was created with the name " + ADDRESS));
        HashSet<RoutingType> routingTypes = new HashSet<RoutingType>();
        routingTypes.add(RoutingType.valueOf((String)param));
        Assertions.assertEquals(routingTypes, (Object)addressInfo.getRoutingTypes());
        conn.disconnect();
    }

    @TestTemplate
    public void testPrefixedAutoCreatedAnycastAndMulticastWithSameName() throws Exception {
        int port = 61614;
        URI uri = StompTest.createStompClientUri(this.scheme, this.hostname, port);
        String ADDRESS = UUID.randomUUID().toString();
        this.server.getRemotingService().createAcceptor("test", "tcp://" + this.hostname + ":" + port + "?protocols=STOMP&anycastPrefix=/queue/&multicastPrefix=/topic/").start();
        StompClientConnection conn = StompClientConnectionFactory.createClientConnection(uri);
        conn.connect(this.defUser, this.defPass);
        String uuid = UUID.randomUUID().toString();
        ClientStompFrame frame = conn.createFrame("SUBSCRIBE").addHeader("destination", "/queue/" + ADDRESS).addHeader("receipt", uuid);
        frame = conn.sendFrame(frame);
        Assertions.assertEquals((Object)uuid, (Object)frame.getHeader("receipt-id"));
        AddressInfo addressInfo = this.server.getAddressInfo(SimpleString.of((String)ADDRESS));
        Assertions.assertNotNull((Object)addressInfo, (String)("No address was created with the name " + ADDRESS));
        Assertions.assertTrue((boolean)addressInfo.getRoutingTypes().contains(RoutingType.ANYCAST));
        Assertions.assertFalse((boolean)addressInfo.getRoutingTypes().contains(RoutingType.MULTICAST));
        Assertions.assertNotNull((Object)this.server.locateQueue(SimpleString.of((String)ADDRESS)));
        frame = StompTest.send(conn, "/topic/" + ADDRESS, null, "Hello World 1", true);
        Assertions.assertFalse((boolean)frame.getCommand().equals("ERROR"));
        addressInfo = this.server.getAddressInfo(SimpleString.of((String)ADDRESS));
        Assertions.assertTrue((boolean)addressInfo.getRoutingTypes().contains(RoutingType.ANYCAST));
        Assertions.assertTrue((boolean)addressInfo.getRoutingTypes().contains(RoutingType.MULTICAST));
        frame = conn.receiveFrame(100L);
        Assertions.assertNull((Object)frame);
        frame = StompTest.send(conn, "/queue/" + ADDRESS, null, "Hello World 2", true);
        Assertions.assertFalse((boolean)frame.getCommand().equals("ERROR"));
        frame = conn.receiveFrame(1000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)"Hello World 2", (Object)frame.getBody());
        Assertions.assertEquals((Object)RoutingType.ANYCAST.toString(), (Object)frame.getHeader("destination-type"));
        Assertions.assertEquals((Object)("/queue/" + ADDRESS), (Object)frame.getHeader("destination"));
        frame = conn.receiveFrame(100L);
        Assertions.assertNull((Object)frame);
        StompTest.unsubscribe(conn, null, "/queue/" + ADDRESS, true, false);
        Wait.assertTrue(() -> this.server.getAddressInfo(SimpleString.of((String)ADDRESS)) == null);
        uuid = UUID.randomUUID().toString();
        frame = conn.createFrame("SUBSCRIBE").addHeader("destination", "/topic/" + ADDRESS).addHeader("receipt", uuid);
        frame = conn.sendFrame(frame);
        Assertions.assertEquals((Object)uuid, (Object)frame.getHeader("receipt-id"));
        frame = StompTest.send(conn, "/topic/" + ADDRESS, null, "Hello World 3", true);
        Assertions.assertFalse((boolean)frame.getCommand().equals("ERROR"));
        frame = conn.receiveFrame(1000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)"Hello World 3", (Object)frame.getBody());
        Assertions.assertEquals((Object)RoutingType.MULTICAST.toString(), (Object)frame.getHeader("destination-type"));
        Assertions.assertEquals((Object)("/topic/" + ADDRESS), (Object)frame.getHeader("destination"));
        frame = conn.receiveFrame(100L);
        Assertions.assertNull((Object)frame);
        StompTest.unsubscribe(conn, null, "/topic/" + ADDRESS, true, false);
        conn.disconnect();
    }

    @TestTemplate
    public void testPrefixedAutoCreatedMulticastAndAnycastWithSameName() throws Exception {
        int port = 61614;
        URI uri = StompTest.createStompClientUri(this.scheme, this.hostname, port);
        String ADDRESS = UUID.randomUUID().toString();
        this.server.getRemotingService().createAcceptor("test", "tcp://" + this.hostname + ":" + port + "?protocols=STOMP&anycastPrefix=/queue/&multicastPrefix=/topic/").start();
        StompClientConnection conn = StompClientConnectionFactory.createClientConnection(uri);
        conn.connect(this.defUser, this.defPass);
        String uuid = UUID.randomUUID().toString();
        ClientStompFrame frame = conn.createFrame("SUBSCRIBE").addHeader("destination", "/topic/" + ADDRESS).addHeader("receipt", uuid);
        frame = conn.sendFrame(frame);
        Assertions.assertEquals((Object)uuid, (Object)frame.getHeader("receipt-id"));
        AddressInfo addressInfo = this.server.getAddressInfo(SimpleString.of((String)ADDRESS));
        Assertions.assertNotNull((Object)addressInfo, (String)("No address was created with the name " + ADDRESS));
        Assertions.assertTrue((boolean)addressInfo.getRoutingTypes().contains(RoutingType.MULTICAST));
        Assertions.assertFalse((boolean)addressInfo.getRoutingTypes().contains(RoutingType.ANYCAST));
        frame = StompTest.send(conn, "/queue/" + ADDRESS, null, "Hello World 1", true);
        Assertions.assertFalse((boolean)frame.getCommand().equals("ERROR"));
        addressInfo = this.server.getAddressInfo(SimpleString.of((String)ADDRESS));
        Assertions.assertTrue((boolean)addressInfo.getRoutingTypes().contains(RoutingType.ANYCAST));
        Assertions.assertTrue((boolean)addressInfo.getRoutingTypes().contains(RoutingType.MULTICAST));
        Assertions.assertNotNull((Object)this.server.locateQueue(SimpleString.of((String)ADDRESS)));
        frame = conn.receiveFrame(100L);
        Assertions.assertNull((Object)frame);
        frame = StompTest.send(conn, "/topic/" + ADDRESS, null, "Hello World 2", true);
        Assertions.assertFalse((boolean)frame.getCommand().equals("ERROR"));
        frame = conn.receiveFrame(2000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)"Hello World 2", (Object)frame.getBody());
        Assertions.assertEquals((Object)RoutingType.MULTICAST.toString(), (Object)frame.getHeader("destination-type"));
        Assertions.assertEquals((Object)("/topic/" + ADDRESS), (Object)frame.getHeader("destination"));
        frame = conn.receiveFrame(100L);
        Assertions.assertNull((Object)frame);
        frame = StompTest.unsubscribe(conn, null, "/topic/" + ADDRESS, true, false);
        Assertions.assertFalse((boolean)frame.getCommand().equals("ERROR"));
        uuid = UUID.randomUUID().toString();
        frame = conn.createFrame("SUBSCRIBE").addHeader("destination", "/queue/" + ADDRESS).addHeader("receipt", uuid);
        frame = conn.sendFrame(frame);
        Assertions.assertEquals((Object)uuid, (Object)frame.getHeader("receipt-id"));
        frame = conn.receiveFrame(1000L);
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)"Hello World 1", (Object)frame.getBody());
        Assertions.assertEquals((Object)RoutingType.ANYCAST.toString(), (Object)frame.getHeader("destination-type"));
        Assertions.assertEquals((Object)("/queue/" + ADDRESS), (Object)frame.getHeader("destination"));
        frame = conn.receiveFrame(100L);
        Assertions.assertNull((Object)frame);
        StompTest.unsubscribe(conn, null, "/queue/" + ADDRESS, true, false);
        conn.disconnect();
    }

    @TestTemplate
    public void testDotPrefixedSendAndRecieveAnycast() throws Exception {
        this.testPrefixedSendAndRecieve("jms.queue.", RoutingType.ANYCAST);
    }

    @TestTemplate
    public void testDotPrefixedSendAndRecieveMulticast() throws Exception {
        this.testPrefixedSendAndRecieve("jms.topic.", RoutingType.MULTICAST);
    }

    @TestTemplate
    public void testSlashPrefixedSendAndRecieveAnycast() throws Exception {
        this.testPrefixedSendAndRecieve("/queue/", RoutingType.ANYCAST);
    }

    @TestTemplate
    public void testSlashPrefixedSendAndRecieveMulticast() throws Exception {
        this.testPrefixedSendAndRecieve("/topic/", RoutingType.MULTICAST);
    }

    public void testPrefixedSendAndRecieve(String prefix, RoutingType routingType) throws Exception {
        int port = 61614;
        URI uri = StompTest.createStompClientUri(this.scheme, this.hostname, port);
        String ADDRESS = UUID.randomUUID().toString();
        String PREFIXED_ADDRESS = prefix + ADDRESS;
        String urlParam = routingType.toString().toLowerCase() + "Prefix";
        this.server.getRemotingService().createAcceptor("test", "tcp://" + this.hostname + ":" + port + "?protocols=STOMP&" + urlParam + "=" + prefix).start();
        StompClientConnection conn = StompClientConnectionFactory.createClientConnection(uri);
        conn.connect(this.defUser, this.defPass);
        String uuid = UUID.randomUUID().toString();
        ClientStompFrame frame = conn.createFrame("SUBSCRIBE").addHeader("destination", PREFIXED_ADDRESS).addHeader("receipt", uuid);
        frame = conn.sendFrame(frame);
        Assertions.assertEquals((Object)uuid, (Object)frame.getHeader("receipt-id"));
        StompTest.send(conn, ADDRESS, null, "Hello World", true);
        frame = conn.receiveFrame(10000L);
        Assertions.assertNotNull((Object)frame, (String)"Should have received a message");
        Assertions.assertEquals((Object)"MESSAGE", (Object)frame.getCommand());
        Assertions.assertEquals((Object)ADDRESS, (Object)frame.getHeader("destination"));
        Assertions.assertEquals((Object)"Hello World", (Object)frame.getBody());
        conn.disconnect();
    }

    @TestTemplate
    public void testMulticastOperationsOnAnycastAddress() throws Exception {
        this.server.getAddressSettingsRepository().addMatch("#", (Object)new AddressSettings().setAutoCreateAddresses(Boolean.valueOf(false)).setAutoCreateQueues(Boolean.valueOf(false)));
        this.testRoutingSemantics(StompTest.getQueuePrefix() + StompTest.getQueueName(), RoutingType.MULTICAST.toString());
    }

    @TestTemplate
    public void testAnycastOperationsOnMulticastAddress() throws Exception {
        this.server.getAddressSettingsRepository().addMatch("#", (Object)new AddressSettings().setAutoCreateAddresses(Boolean.valueOf(false)).setAutoCreateQueues(Boolean.valueOf(false)));
        this.testRoutingSemantics(StompTest.getTopicPrefix() + StompTest.getTopicName(), RoutingType.ANYCAST.toString());
    }

    public void testRoutingSemantics(String routingType, String destination) throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        String uuid = UUID.randomUUID().toString();
        ClientStompFrame frame = this.conn.createFrame("SUBSCRIBE").addHeader("subscription-type", routingType).addHeader("destination", destination).addHeader("receipt", uuid);
        frame = this.conn.sendFrame(frame);
        Assertions.assertEquals((Object)"ERROR", (Object)frame.getCommand());
        uuid = UUID.randomUUID().toString();
        frame = this.conn.createFrame("SEND").addHeader("destination-type", RoutingType.MULTICAST.toString()).addHeader("destination", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader("receipt", uuid);
        frame = this.conn.sendFrame(frame);
        Assertions.assertEquals((Object)"ERROR", (Object)frame.getCommand());
    }

    @TestTemplate
    public void testGetManagementAttributeFromStomp() throws Exception {
        this.server.getAddressSettingsRepository().addMatch("#", (Object)new AddressSettings().setAutoCreateAddresses(Boolean.valueOf(false)).setAutoCreateQueues(Boolean.valueOf(false)));
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", ActiveMQDefaultConfiguration.getDefaultManagementAddress().toString()).addHeader("reply-to", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader(ManagementHelper.HDR_RESOURCE_NAME.toString(), "queue." + StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader(ManagementHelper.HDR_ATTRIBUTE.toString(), "Address");
        this.conn.sendFrame(frame);
        frame = this.conn.receiveFrame(10000L);
        logger.debug("Received: {}", (Object)frame);
        Assertions.assertEquals((Object)Boolean.TRUE.toString(), (Object)frame.getHeader(ManagementHelper.HDR_OPERATION_SUCCEEDED.toString()));
        Assertions.assertEquals((Object)("[\"" + StompTest.getQueuePrefix() + StompTest.getQueueName() + "\"]"), (Object)frame.getBody());
        StompTest.unsubscribe(this.conn, null);
        this.conn.disconnect();
    }

    @TestTemplate
    public void testInvokeOperationFromStomp() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null);
        ClientStompFrame frame = this.conn.createFrame("SEND").addHeader("destination", ActiveMQDefaultConfiguration.getDefaultManagementAddress().toString()).addHeader("reply-to", StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader(ManagementHelper.HDR_RESOURCE_NAME.toString(), "queue." + StompTest.getQueuePrefix() + StompTest.getQueueName()).addHeader(ManagementHelper.HDR_OPERATION_NAME.toString(), "countMessages").setBody("[\"color = 'blue'\"]");
        this.conn.sendFrame(frame);
        frame = this.conn.receiveFrame(10000L);
        logger.debug("Received: {}", (Object)frame);
        Assertions.assertEquals((Object)Boolean.TRUE.toString(), (Object)frame.getHeader(ManagementHelper.HDR_OPERATION_SUCCEEDED.toString()));
        Assertions.assertEquals((Object)"[0]", (Object)frame.getBody());
        StompTest.unsubscribe(this.conn, null);
        this.conn.disconnect();
    }

    @TestTemplate
    public void testAnycastMessageRoutingExclusivity() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        String addressA = "addressA";
        String queueA = "queueA";
        String queueB = "queueB";
        String queueC = "queueC";
        ActiveMQServerControlImpl serverControl = this.server.getActiveMQServerControl();
        serverControl.createAddress("addressA", RoutingType.ANYCAST.toString() + "," + RoutingType.MULTICAST.toString());
        serverControl.createQueue(QueueConfiguration.of((String)"queueA").setAddress("addressA").setRoutingType(RoutingType.ANYCAST).toJSON());
        serverControl.createQueue(QueueConfiguration.of((String)"queueB").setAddress("addressA").setRoutingType(RoutingType.ANYCAST).toJSON());
        serverControl.createQueue(QueueConfiguration.of((String)"queueC").setAddress("addressA").setRoutingType(RoutingType.MULTICAST).toJSON());
        StompTest.send(this.conn, "addressA", null, "Hello World!", true, RoutingType.ANYCAST);
        Wait.assertTrue(() -> this.server.locateQueue(SimpleString.of((String)"queueA")).getMessageCount() + this.server.locateQueue(SimpleString.of((String)"queueB")).getMessageCount() == 1L);
        Wait.assertTrue(() -> this.server.locateQueue(SimpleString.of((String)"queueC")).getMessageCount() == 0L);
    }

    @TestTemplate
    public void testMulticastMessageRoutingExclusivity() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        String addressA = "addressA";
        String queueA = "queueA";
        String queueB = "queueB";
        String queueC = "queueC";
        ActiveMQServerControlImpl serverControl = this.server.getActiveMQServerControl();
        serverControl.createAddress("addressA", RoutingType.ANYCAST.toString() + "," + RoutingType.MULTICAST.toString());
        serverControl.createQueue(QueueConfiguration.of((String)"queueA").setAddress("addressA").setRoutingType(RoutingType.ANYCAST).toJSON());
        serverControl.createQueue(QueueConfiguration.of((String)"queueB").setAddress("addressA").setRoutingType(RoutingType.MULTICAST).toJSON());
        serverControl.createQueue(QueueConfiguration.of((String)"queueC").setAddress("addressA").setRoutingType(RoutingType.MULTICAST).toJSON());
        StompTest.send(this.conn, "addressA", null, "Hello World!", true, RoutingType.MULTICAST);
        Wait.assertTrue(() -> this.server.locateQueue(SimpleString.of((String)"queueA")).getMessageCount() == 0L);
        Wait.assertTrue(() -> this.server.locateQueue(SimpleString.of((String)"queueC")).getMessageCount() + this.server.locateQueue(SimpleString.of((String)"queueB")).getMessageCount() == 2L);
    }

    @TestTemplate
    public void testAmbiguousMessageRouting() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        String addressA = "addressA";
        String queueA = "queueA";
        String queueB = "queueB";
        String queueC = "queueC";
        String queueD = "queueD";
        ActiveMQServerControlImpl serverControl = this.server.getActiveMQServerControl();
        serverControl.createAddress("addressA", RoutingType.ANYCAST.toString() + "," + RoutingType.MULTICAST.toString());
        serverControl.createQueue(QueueConfiguration.of((String)"queueA").setAddress("addressA").setRoutingType(RoutingType.ANYCAST).toJSON());
        serverControl.createQueue(QueueConfiguration.of((String)"queueB").setAddress("addressA").setRoutingType(RoutingType.ANYCAST).toJSON());
        serverControl.createQueue(QueueConfiguration.of((String)"queueC").setAddress("addressA").setRoutingType(RoutingType.MULTICAST).toJSON());
        serverControl.createQueue(QueueConfiguration.of((String)"queueD").setAddress("addressA").setRoutingType(RoutingType.MULTICAST).toJSON());
        StompTest.send(this.conn, "addressA", null, "Hello World!", true);
        Wait.assertTrue(() -> this.server.locateQueue(SimpleString.of((String)"queueA")).getMessageCount() + this.server.locateQueue(SimpleString.of((String)"queueB")).getMessageCount() == 1L);
        Wait.assertTrue(() -> this.server.locateQueue(SimpleString.of((String)"queueC")).getMessageCount() + this.server.locateQueue(SimpleString.of((String)"queueD")).getMessageCount() == 2L);
    }

    @TestTemplate
    public void testAutoCreatedAnycastAddress() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        String queueName = UUID.randomUUID().toString();
        SimpleString simpleQueueName = SimpleString.of((String)queueName);
        Assertions.assertNull((Object)this.server.getAddressInfo(simpleQueueName));
        Assertions.assertNull((Object)this.server.locateQueue(simpleQueueName));
        this.server.getAddressSettingsRepository().addMatch(queueName, (Object)new AddressSettings().setDefaultAddressRoutingType(RoutingType.ANYCAST).setDefaultQueueRoutingType(RoutingType.ANYCAST));
        StompTest.send(this.conn, queueName, null, "Hello ANYCAST");
        Wait.assertTrue((String)"Address and queue should be created now", () -> this.server.getAddressInfo(simpleQueueName) != null && this.server.locateQueue(simpleQueueName) != null);
        Assertions.assertTrue((boolean)this.server.getAddressInfo(simpleQueueName).getRoutingTypes().contains(RoutingType.ANYCAST));
        Assertions.assertEquals((Object)RoutingType.ANYCAST, (Object)this.server.locateQueue(simpleQueueName).getRoutingType());
    }

    @TestTemplate
    public void testAutoCreatedMulticastAddress() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        String queueName = UUID.randomUUID().toString();
        SimpleString simpleQueueName = SimpleString.of((String)queueName);
        Assertions.assertNull((Object)this.server.getAddressInfo(simpleQueueName));
        Assertions.assertNull((Object)this.server.locateQueue(simpleQueueName));
        StompTest.send(this.conn, queueName, null, "Hello MULTICAST");
        Wait.assertTrue((String)"Address should be created now", () -> this.server.getAddressInfo(simpleQueueName) != null);
        Assertions.assertTrue((boolean)this.server.getAddressInfo(simpleQueueName).getRoutingTypes().contains(RoutingType.MULTICAST));
        Assertions.assertNull((Object)this.server.locateQueue(simpleQueueName));
    }

    @TestTemplate
    public void directDeliverDisabledOnStomp() throws Exception {
        String payload = "This is a test message";
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribe(this.conn, null, "auto");
        for (Binding b : IterableStream.iterableOf(this.server.getPostOffice().getAllBindings().filter(QueueBinding.class::isInstance))) {
            Assertions.assertFalse((boolean)((QueueBinding)b).getQueue().isDirectDeliver(), (String)("Queue " + ((QueueBinding)b).getQueue().getName()));
        }
        FuseMQTTClientProvider clientProvider = new FuseMQTTClientProvider();
        clientProvider.connect("tcp://" + this.hostname + ":61613");
        clientProvider.publish(StompTest.getQueuePrefix() + StompTest.getQueueName(), payload.getBytes(), 0);
        clientProvider.disconnect();
        ClientStompFrame frame = this.conn.receiveFrame();
        Assertions.assertTrue((boolean)frame.getBody().contains(payload));
    }

    @TestTemplate
    public void testSameMessageHasDifferentMessageIdPerConsumer() throws Exception {
        this.conn.connect(this.defUser, this.defPass);
        StompTest.subscribeTopic(this.conn, "sub1", "client", null);
        StompTest.subscribeTopic(this.conn, "sub2", "client", null);
        this.sendJmsMessage(this.getName(), (Destination)this.topic);
        ClientStompFrame frame1 = this.conn.receiveFrame();
        String firstMessageID = frame1.getHeader("message-id");
        Assertions.assertNotNull((Object)firstMessageID);
        ClientStompFrame frame2 = this.conn.receiveFrame();
        String secondMessageID = frame2.getHeader("message-id");
        Assertions.assertNotNull((Object)secondMessageID);
        Assertions.assertTrue((!firstMessageID.equals(secondMessageID) ? (byte)1 : 0) != 0, (String)(firstMessageID + " must not equal " + secondMessageID));
        StompTest.ack(this.conn, "sub1", frame1);
        StompTest.ack(this.conn, "sub2", frame2);
        StompTest.unsubscribe(this.conn, "sub1");
        StompTest.unsubscribe(this.conn, "sub2");
        this.conn.disconnect();
    }
}

